import { switchExpression } from "@kie-tools-core/switch-expression-ts";
import { snapBoundsDimensions, snapBoundsPosition } from "../SnapGrid";
import { PositionalNodeHandleId } from "../connections/PositionalNodeHandles";
import { AutoPositionedEdgeMarker } from "../edges/AutoPositionedEdgeMarker";
import { NODE_TYPES } from "../nodes/NodeTypes";
import { getCenter } from "./Maths";
export const DEFAULT_INTRACTION_WIDTH = 40;
export const CONTAINER_NODES_DESIRABLE_PADDING = 60;
export function getDistance(a, b) {
    return Math.sqrt(Math.pow(a["@_x"] - b["@_x"], 2) + Math.pow(a["@_y"] - b["@_y"], 2));
}
export function getDmnBoundsCenterPoint(bounds) {
    const { x, y } = getCenter(bounds["@_x"], bounds["@_y"], bounds["@_width"], bounds["@_height"]);
    return { "@_x": x, "@_y": y };
}
export function getPointForHandle({ handle, bounds, }) {
    if (handle === PositionalNodeHandleId.Center) {
        return getDmnBoundsCenterPoint(bounds);
    }
    else if (handle === PositionalNodeHandleId.Top) {
        return { "@_x": bounds["@_x"] + bounds["@_width"] / 2, "@_y": bounds["@_y"] };
    }
    else if (handle === PositionalNodeHandleId.Right) {
        return { "@_x": bounds["@_x"] + bounds["@_width"], "@_y": bounds["@_y"] + bounds["@_height"] / 2 };
    }
    else if (handle === PositionalNodeHandleId.Bottom) {
        return { "@_x": bounds["@_x"] + bounds["@_width"] / 2, "@_y": bounds["@_y"] + bounds["@_height"] };
    }
    else if (handle === PositionalNodeHandleId.Left) {
        return { "@_x": bounds["@_x"], "@_y": bounds["@_y"] + bounds["@_height"] / 2 };
    }
    else {
        throw new Error(`Invalid target handle id '${handle}'.`);
    }
}
export function getHandlePosition({ shapeBounds, waypoint, }) {
    var _a, _b, _c, _d;
    const x = (_a = shapeBounds === null || shapeBounds === void 0 ? void 0 : shapeBounds["@_x"]) !== null && _a !== void 0 ? _a : 0;
    const y = (_b = shapeBounds === null || shapeBounds === void 0 ? void 0 : shapeBounds["@_y"]) !== null && _b !== void 0 ? _b : 0;
    const w = (_c = shapeBounds === null || shapeBounds === void 0 ? void 0 : shapeBounds["@_width"]) !== null && _c !== void 0 ? _c : 0;
    const h = (_d = shapeBounds === null || shapeBounds === void 0 ? void 0 : shapeBounds["@_height"]) !== null && _d !== void 0 ? _d : 0;
    const center = { "@_x": x + w / 2, "@_y": y + h / 2 };
    const left = { "@_x": x, "@_y": y + h / 2 };
    const right = { "@_x": x + w, "@_y": y + h / 2 };
    const top = { "@_x": x + w / 2, "@_y": y };
    const bottom = { "@_x": x + w / 2, "@_y": y + h };
    if (getDistance(center, waypoint) <= 1) {
        return { handlePosition: PositionalNodeHandleId.Center, point: center };
    }
    else if (getDistance(top, waypoint) <= 1) {
        return { handlePosition: PositionalNodeHandleId.Top, point: top };
    }
    else if (getDistance(right, waypoint) <= 1) {
        return { handlePosition: PositionalNodeHandleId.Right, point: right };
    }
    else if (getDistance(bottom, waypoint) <= 1) {
        return { handlePosition: PositionalNodeHandleId.Bottom, point: bottom };
    }
    else if (getDistance(left, waypoint) <= 1) {
        return { handlePosition: PositionalNodeHandleId.Left, point: left };
    }
    else {
        console.warn("DMN DIAGRAM: Can't find a match of NSWE/Center handles. Using Center as default.");
        return { handlePosition: PositionalNodeHandleId.Center, point: center };
    }
}
export function getLineRectangleIntersectionPoint(point1, point2, rectangle) {
    const [x1, y1] = [point1["@_x"], point1["@_y"]];
    const [x2, y2] = [point2["@_x"], point2["@_y"]];
    const [rx, ry] = [rectangle.x, rectangle.y];
    const [rw, rh] = [rectangle.width, rectangle.height];
    const m = (y2 - y1) / (x2 - x1);
    const b = y1 - m * x1;
    if (m === Infinity || m === -Infinity) {
        const x = point1["@_x"];
        const minY = Math.min(point1["@_y"], point2["@_y"]);
        const maxY = Math.max(point1["@_y"], point2["@_y"]);
        if (x >= rectangle.x && x <= rectangle.x + rectangle.width) {
            if (minY <= rectangle.y) {
                return { "@_x": x, "@_y": rectangle.y };
            }
            else if (maxY >= rectangle.y + rectangle.height) {
                return { "@_x": x, "@_y": rectangle.y + rectangle.height };
            }
        }
    }
    const intersections = [];
    const topX = Math.round((ry - b) / m);
    if (topX >= rx && topX <= rx + rw) {
        intersections.push({ x: topX, y: ry });
    }
    const bottomX = Math.round((ry + rh - b) / m);
    if (bottomX >= rx && bottomX <= rx + rw) {
        intersections.push({ x: bottomX, y: ry + rh });
    }
    const leftY = Math.round(m * rx + b);
    if (leftY >= ry && leftY <= ry + rh) {
        intersections.push({ x: rx, y: leftY });
    }
    const rightY = Math.round(m * (rx + rw) + b);
    if (rightY >= ry && rightY <= ry + rh) {
        intersections.push({ x: rx + rw, y: rightY });
    }
    let closestIntersection = null;
    for (const intersection of intersections) {
        if (!closestIntersection || minDistance(intersection, x1, y1) < minDistance(closestIntersection, x1, y1)) {
            closestIntersection = intersection;
        }
    }
    return ((closestIntersection && {
        "@_x": closestIntersection.x,
        "@_y": closestIntersection.y,
    }) ||
        point2);
}
const minDistance = (point, x1, y1) => Math.pow(point.x - x1, 2) + Math.pow(point.y - y1, 2);
export function getContainmentRelationship({ bounds, container, divingLineLocalY, snapGrid, isAlternativeInputDataShape, containerMinSizes, boundsMinSizes, }) {
    const { x: cx, y: cy } = snapBoundsPosition(snapGrid, container);
    const { width: cw, height: ch } = snapBoundsDimensions(snapGrid, container, containerMinSizes({ snapGrid, isAlternativeInputDataShape }));
    const { x: bx, y: by } = snapBoundsPosition(snapGrid, bounds);
    const { width: bw, height: bh } = snapBoundsDimensions(snapGrid, bounds, boundsMinSizes({ snapGrid, isAlternativeInputDataShape }));
    const center = getDmnBoundsCenterPoint({
        "@_height": bh,
        "@_width": bw,
        "@_x": bx,
        "@_y": by,
    });
    const isInside = bx >= cx &&
        by >= cy &&
        bx + bw <= cx + cw &&
        by + bh <= cy + ch;
    if (isInside) {
        return { isInside: true, section: center["@_y"] > cy + (divingLineLocalY !== null && divingLineLocalY !== void 0 ? divingLineLocalY : 0) ? "lower" : "upper" };
    }
    else {
        return { isInside: false };
    }
}
export function pointsToPath(points) {
    const start = points[0];
    let path = `M ${start["@_x"]},${start["@_y"]}`;
    for (let i = 1; i < points.length - 1; i++) {
        const p = points[i];
        path += ` L ${p["@_x"]},${p["@_y"]} M ${p["@_x"]},${p["@_y"]}`;
    }
    const end = points[points.length - 1];
    path += ` L ${end["@_x"]},${end["@_y"]}`;
    return path;
}
export function getDecisionServiceDividerLineLocalY(shape) {
    var _a, _b, _c, _d, _e;
    return (((_c = (_b = (_a = shape["dmndi:DMNDecisionServiceDividerLine"]) === null || _a === void 0 ? void 0 : _a["di:waypoint"]) === null || _b === void 0 ? void 0 : _b[0]["@_y"]) !== null && _c !== void 0 ? _c : 0) -
        ((_e = (_d = shape["dc:Bounds"]) === null || _d === void 0 ? void 0 : _d["@_y"]) !== null && _e !== void 0 ? _e : 0));
}
export const DISCRETE_AUTO_POSITIONING_DMN_EDGE_ID_MARKER = [
    AutoPositionedEdgeMarker.BOTH,
    AutoPositionedEdgeMarker.SOURCE,
    AutoPositionedEdgeMarker.TARGET,
];
export function getDiscreteAutoPositioningEdgeIdMarker(edgeId) {
    for (const marker of DISCRETE_AUTO_POSITIONING_DMN_EDGE_ID_MARKER) {
        if (edgeId.endsWith(marker)) {
            return marker;
        }
    }
    return undefined;
}
export function getBounds({ nodes, padding, }) {
    var _a, _b;
    let maxX = 0, maxY = 0, minX = Infinity, minY = Infinity;
    for (let i = 0; i < nodes.length; i++) {
        const node = nodes[i];
        maxX = Math.max(maxX, node.position.x + ((_a = node.width) !== null && _a !== void 0 ? _a : 0));
        minX = Math.min(minX, node.position.x);
        maxY = Math.max(maxY, node.position.y + ((_b = node.height) !== null && _b !== void 0 ? _b : 0));
        minY = Math.min(minY, node.position.y);
    }
    return {
        "@_x": minX - padding,
        "@_y": minY - padding,
        "@_width": maxX - minX + 2 * padding,
        "@_height": maxY - minY + 2 * padding,
    };
}
export function getNodeTypeFromDmnObject(dmnObject) {
    if (!dmnObject) {
        return NODE_TYPES.unknown;
    }
    const type = switchExpression(dmnObject.__$$element, {
        inputData: NODE_TYPES.inputData,
        decision: NODE_TYPES.decision,
        businessKnowledgeModel: NODE_TYPES.bkm,
        knowledgeSource: NODE_TYPES.knowledgeSource,
        decisionService: NODE_TYPES.decisionService,
        group: NODE_TYPES.group,
        textAnnotation: NODE_TYPES.textAnnotation,
        default: undefined,
    });
    return type;
}
//# sourceMappingURL=DmnMaths.js.map