/*
 * Decompiled with CFR 0.152.
 */
package org.archive.io;

import java.io.File;
import java.io.IOException;
import java.util.Collection;
import java.util.LinkedList;
import java.util.concurrent.ArrayBlockingQueue;
import java.util.concurrent.BlockingQueue;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.logging.Level;
import java.util.logging.Logger;
import org.archive.io.WriterPoolMember;
import org.archive.io.WriterPoolSettings;
import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;

public abstract class WriterPool {
    private final Logger logger = Logger.getLogger(this.getClass().getName());
    protected final AtomicInteger serialNo;
    public static final int DEFAULT_MAX_ACTIVE = 1;
    protected static final int LARGEST_MAX_ACTIVE = 255;
    public static final int DEFAULT_MAX_WAIT_FOR_IDLE = 500;
    protected final WriterPoolSettings settings;
    protected int maxActive;
    protected int maxWait;
    protected int currentActive = 0;
    protected BlockingQueue<WriterPoolMember> availableWriters;
    protected long lastWriterNeededTime;
    protected long lastWriterRolloverTime;

    public WriterPool(AtomicInteger serial, WriterPoolSettings settings, int poolMaximumActive, int poolMaximumWait) {
        this.logger.info("Initial configuration: prefix=" + settings.getPrefix() + ", template=" + settings.getTemplate() + ", compress=" + settings.getCompress() + ", maxSize=" + settings.getMaxFileSizeBytes() + ", maxActive=" + poolMaximumActive + ", maxWait=" + poolMaximumWait);
        this.settings = settings;
        this.maxActive = poolMaximumActive;
        this.maxWait = poolMaximumWait;
        this.availableWriters = new ArrayBlockingQueue<WriterPoolMember>(255, true);
        this.serialNo = serial;
    }

    public WriterPoolMember borrowFile() throws IOException {
        WriterPoolMember writer = null;
        while (writer == null) {
            try {
                writer = this.availableWriters.poll(this.maxWait, TimeUnit.MILLISECONDS);
            }
            catch (InterruptedException interruptedException) {
                // empty catch block
            }
            if (writer != null) continue;
            writer = this.makeNewWriterIfAppropriate();
        }
        return writer;
    }

    protected synchronized WriterPoolMember makeNewWriterIfAppropriate() {
        long now;
        this.lastWriterNeededTime = now = System.currentTimeMillis();
        if (this.currentActive < this.maxActive) {
            ++this.currentActive;
            this.lastWriterRolloverTime = now;
            return this.makeWriter();
        }
        return null;
    }

    protected abstract WriterPoolMember makeWriter();

    public synchronized void destroyWriter(WriterPoolMember writer) throws IOException {
        --this.currentActive;
        writer.close();
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void returnFile(WriterPoolMember writer) throws IOException {
        WriterPool writerPool = this;
        synchronized (writerPool) {
            if (writer.isOversize()) {
                if (this.lastWriterNeededTime <= this.lastWriterRolloverTime) {
                    this.destroyWriter(writer);
                    return;
                }
                this.lastWriterRolloverTime = System.currentTimeMillis();
            }
        }
        if (!this.availableWriters.offer(writer)) {
            this.logger.log(Level.WARNING, "writer unreturnable to available pool; closing early");
            this.destroyWriter(writer);
        }
    }

    public synchronized void invalidateFile(WriterPoolMember f) throws IOException {
        try {
            this.destroyWriter(f);
        }
        catch (Exception e) {
            throw new IOException(e.getMessage());
        }
        File file = f.getFile();
        file.renameTo(new File(file.getAbsoluteFile() + ".invalid"));
    }

    public synchronized int getNumActive() throws UnsupportedOperationException {
        return this.currentActive - this.getNumIdle();
    }

    public int getNumIdle() throws UnsupportedOperationException {
        return this.availableWriters.size();
    }

    public void close() {
        Collection<WriterPoolMember> writers = this.drainAllWriters();
        for (WriterPoolMember writer : writers) {
            try {
                this.destroyWriter(writer);
            }
            catch (IOException e) {
                this.logger.log(Level.WARNING, "problem closing writer", e);
            }
        }
    }

    public WriterPoolSettings getSettings() {
        return this.settings;
    }

    protected String getPoolState() {
        StringBuffer buffer = new StringBuffer("Active ");
        buffer.append(this.getNumActive());
        buffer.append(" of max ");
        buffer.append(this.maxActive);
        buffer.append(", idle ");
        buffer.append(this.getNumIdle());
        return buffer.toString();
    }

    public AtomicInteger getSerialNo() {
        return this.serialNo;
    }

    protected synchronized Collection<WriterPoolMember> drainAllWriters() {
        LinkedList<WriterPoolMember> writers = new LinkedList<WriterPoolMember>();
        this.availableWriters.drainTo(writers);
        while (writers.size() < this.currentActive) {
            try {
                WriterPoolMember w = this.availableWriters.take();
                writers.add(w);
            }
            catch (InterruptedException e) {
                this.logger.severe("caught " + e + " while waiting for writers to free up; returning only " + writers.size() + " of " + this.currentActive + " active writers");
                break;
            }
        }
        return writers;
    }

    public void flush() {
        Collection<WriterPoolMember> writers = this.drainAllWriters();
        for (WriterPoolMember writer : writers) {
            try {
                writer.flush();
            }
            catch (IOException e) {
                this.logger.log(Level.WARNING, "problem flushing writer " + writer, e);
            }
        }
        this.availableWriters.addAll(writers);
    }

    public JSONArray jsonStatus() throws JSONException {
        Collection<WriterPoolMember> writers = this.drainAllWriters();
        JSONArray ja = new JSONArray();
        for (WriterPoolMember w : writers) {
            JSONObject jo = new JSONObject();
            jo.put("file", (Object)w.getFile());
            jo.put("position", w.getPosition());
            ja.put((Object)jo);
        }
        this.availableWriters.addAll(writers);
        return ja;
    }
}

